package com.taiwanmobile.pt.tamedia.tutorial.compose

import android.graphics.Color
import android.os.Bundle
import android.view.LayoutInflater
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.SystemBarStyle
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.navigationBars
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.windowInsetsBottomHeight
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import com.taiwanmobile.pt.adp.nativead.TWMMediaContent.ImageSize
import com.taiwanmobile.pt.adp.nativead.TWMNativeAdOptions
import com.taiwanmobile.pt.adp.view.TWMAd
import com.taiwanmobile.pt.adp.view.TWMAdRequest
import com.taiwanmobile.pt.adp.view.TWMAdViewListener
import com.taiwanmobile.pt.adp.view.TWMNativeAd
import com.taiwanmobile.pt.tamedia.tutorial.databinding.ActivityNativeBinding

const val nativeAdUnitId = "YOUR_NATIVE_AD_UNIT_ID"

class NativeAdComposeActivity : ComponentActivity() {

    companion object {
        const val KEY_PREFER_IMAGE: String = "KEY_PREFER_IMAGE"
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge(
            statusBarStyle = SystemBarStyle.light(
                Color.TRANSPARENT, Color.TRANSPARENT
            )
        )

        val preferImage = intent.getBooleanExtra(KEY_PREFER_IMAGE, false)

        setContent {

            val context = LocalContext.current
            var adReceived by remember { mutableStateOf(false) }
            var nativeAd by remember { mutableStateOf<TWMNativeAd?>(null) }

            DisposableEffect(Unit) {
                nativeAd = TWMNativeAd(context, nativeAdUnitId).apply {
                    setAdListener(object: TWMAdViewListener {
                        override fun onReceiveAd(ad: TWMAd) {
                            adReceived = true
                        }

                        override fun onFailedToReceiveAd(
                            ad: TWMAd,
                            errorCode: TWMAdRequest.ErrorCode
                        ) {
                            adReceived = false
                            Toast.makeText(
                                context,
                                errorCode.name,
                                Toast.LENGTH_SHORT
                            ).show()                        }

                        override fun onPresentScreen(ad: TWMAd) {}
                        override fun onDismissScreen(ad: TWMAd) {}
                        override fun onLeaveApplication(ad: TWMAd) {}
                    })
                }

                val adRequest = TWMAdRequest().apply {

                    val disableImageLoading = false
                    val videoStartUnmuted = false
                    val videoCustomControlsRequested = false

                    // setup TWMNativeAd options
                    val nativeOptions: MutableList<TWMNativeAdOptions?> = ArrayList()
                    // speed up the nativeAd loading (optional)
                    if (disableImageLoading) {
                        nativeOptions.add(TWMNativeAdOptions.DISABLE_IMAGE_LOADING)
                    }
                    // prefer video sound is unmuted, default is video sound is muted
                    if (videoStartUnmuted) {
                        nativeOptions.add(TWMNativeAdOptions.VIDEO_START_UNMUTED)
                    }
                    // prefer custom controls, no default media UI control
                    if (videoCustomControlsRequested) {
                        nativeOptions.add(TWMNativeAdOptions.VIDEO_CUSTOM_CONTROLS_REQUESTED)
                    }
                    // prefer display image, default content is video
                    if (preferImage) {
                        nativeOptions.add(TWMNativeAdOptions.MEDIA_PREFER_IMAGE)
                    }

                    if (nativeOptions.isNotEmpty()) {
                        val nativeAdOptionsArray = nativeOptions.filterNotNull().toTypedArray()
                        setNativeAdOptions(nativeAdOptionsArray)
                    }
                }
                nativeAd?.loadAd(adRequest)

                onDispose {
                    nativeAd?.destroy()
                }
            }

            HomeScreen(
                adReceived = adReceived,
                adUnitId = nativeAdUnitId,
                nativeAd = nativeAd
            )
        }
    }

}

@Suppress("SameParameterValue")
@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun HomeScreen(
    adReceived: Boolean,
    adUnitId: String,
    nativeAd: TWMNativeAd?) {

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text("NativeAd Example") }
            )
        },
        bottomBar = {
            Box(
                modifier = Modifier
                    .fillMaxWidth()
                    .windowInsetsBottomHeight(WindowInsets.navigationBars)
                    .background(MaterialTheme.colorScheme.primaryContainer)
            )
        }
    ) { innerPadding ->

        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(innerPadding)
                .padding(horizontal = 16.dp),
            verticalArrangement = Arrangement.Top,
            horizontalAlignment = Alignment.Start
        ) {
            if (adReceived) {
                Text("AdUnitId: $adUnitId")

                Spacer(modifier = Modifier.height(16.dp))

                nativeAd?.let {
                    TWMNativeAdComposable(nativeAd = it)
                }
            }
        }
    }
}

/**
 * Construct TWMNativeView from TWMNativeAd.
 * Use ViewBinding to inflate View-based XML and set view properties.
 *
 * @param nativeAd [TWMNativeAd] to construct TWMNativeView
 * @param modifier [Modifier]
 */
@Composable
fun TWMNativeAdComposable(
    nativeAd: TWMNativeAd,
    modifier: Modifier = Modifier
) {
    AndroidView(
        modifier = modifier.fillMaxWidth(),
        factory = { context ->
            ActivityNativeBinding.inflate(LayoutInflater.from(context)).apply {
                nativeAd.nativeAdContent.run {
                    // short subject
                    shortSubject?.let {
                        nativeAdView.setShortSubjectView(nativeShortTitle.apply { text = it })
                    }
                    // long subject
                    longSubject?.let {
                        nativeAdView.setLongSubjectView(nativeLongTitle.apply { text = it })
                    }
                    // body
                    body?.let {
                        nativeAdView.setBodyView(nativeContent.apply { text = it })
                    }
                    // square icon
                    iconSquare?.let {
                        nativeImage120x120.setImageDrawable(it.getDrawable())
                        nativeImage120x120.adjustViewBounds = true
                        nativeAdView.setSquareIconView(nativeImage120x120)
                    }
                    // rectangle icon
                    iconRect?.let {
                        nativeImage96x64.setImageDrawable(it.getDrawable())
                        nativeImage96x64.adjustViewBounds = true
                        nativeAdView.setRectangleIconView(nativeImage96x64)
                    }
                    // cta
                    callToAction?.let {
                        nativeAdView.setCallToActionView(nativeNativeCta.apply { text = it })
                    }
                    // image 960x640 (non-clickable)
                    image960_640?.let {
                        nativeImage960640.setImageDrawable(it.getDrawable())
                        nativeImage960640.adjustViewBounds = true
                    }
                    // image 1200x627 (non-clickable)
                    image1200_627?.let {
                        nativeImage1200627.setImageDrawable(it.getDrawable())
                        nativeImage1200627.adjustViewBounds = true
                    }
                    // mediaView
                    mediaContent?.let {
                        // if native ad contains no video, specify the preferred image size (BIG or SMALL)
                        if (!it.hasVideoContent()) {
                            it.setMainImageSize(ImageSize.SMALL)
                        }
                        nativeAdView.setMediaView(nativeMediaView.apply { setMediaContent(it) })
                    }
                }
                // specify TWMNativeAd to TWMNativeAdView
                nativeAdView.setNativeAd(nativeAd)
            }.root
        }
    )
}
