package com.taiwanmobile.pt.tamedia.tutorial.compose

import android.app.Activity
import android.content.ContextWrapper
import android.graphics.Color
import android.os.Bundle
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.SystemBarStyle
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.navigationBars
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.windowInsetsBottomHeight
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import com.taiwanmobile.pt.adp.view.TWMAd
import com.taiwanmobile.pt.adp.view.TWMAdRequest
import com.taiwanmobile.pt.adp.view.TWMAdSize
import com.taiwanmobile.pt.adp.view.TWMAdView
import com.taiwanmobile.pt.adp.view.TWMAdViewListener

sealed class BannerAdResult {
    object OK : BannerAdResult()
    data class Error(val message: String) : BannerAdResult()
}

enum class BannerAdType(val adSize: TWMAdSize, val adUnitId: String) {
    BANNER_320X50(TWMAdSize.BANNER, "YOUR_BANNER_AD_UNIT_ID"),
    BANNER_300X250(TWMAdSize.BANNER_300X250, "YOUR_BANNER_AD_UNIT_ID"),
    BANNER_1200X627(TWMAdSize.BANNER_1200X627, "YOUR_BANNER_AD_UNIT_ID")
}

class BannerComposeActivity : ComponentActivity() {

    companion object {
        const val KEY_BANNER_TYPE: String = "BANNER_TYPE"
        const val TYPE_BANNER_320X50: Int = 0
        const val TYPE_BANNER_300X250: Int = 1
        const val TYPE_BANNER_1200X627: Int = 2
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge(
            statusBarStyle = SystemBarStyle.light(
                Color.TRANSPARENT, Color.TRANSPARENT
            )
        )

        val bannerType = when (intent.getIntExtra(KEY_BANNER_TYPE, TYPE_BANNER_320X50)) {
            TYPE_BANNER_300X250 -> BannerAdType.BANNER_300X250
            TYPE_BANNER_1200X627 -> BannerAdType.BANNER_1200X627
            else -> BannerAdType.BANNER_320X50
        }

        setContent {
            HomeScreen(bannerType)
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun HomeScreen(bannerType: BannerAdType) {

    val context = LocalContext.current

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text("Banner Example") }
            )
        },
        bottomBar = {
            Box(
                modifier = Modifier
                    .fillMaxWidth()
                    .windowInsetsBottomHeight(WindowInsets.navigationBars)
                    .background(MaterialTheme.colorScheme.primaryContainer)
            )
        }
    ) { innerPadding ->

        var adResult: BannerAdResult? by remember { mutableStateOf(null) }

        // Use LaunchedEffect to show a toast only when the adResult changes to an error.
        LaunchedEffect(adResult) {
            val result = adResult
            if (result is BannerAdResult.Error) {
                Toast.makeText(context, result.message, Toast.LENGTH_LONG).show()
            }
        }

        if (adResult is BannerAdResult.OK) {
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(innerPadding)
                    .padding(horizontal = 16.dp),
                verticalArrangement = Arrangement.Top,
                horizontalAlignment = Alignment.Start
            ) {
                Text("AdUnitId: ${bannerType.adUnitId}")
            }

        }

        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(innerPadding),
            verticalArrangement = Arrangement.Bottom,
            horizontalAlignment = Alignment.CenterHorizontally
        ) {
            if (adResult == null || adResult is BannerAdResult.OK) {
                TWMBannerComposable(bannerType = bannerType) { result ->
                    adResult = result
                }
            }
        }
    }
}

/**
 * Initialize TWMAdView by specifying the [BannerAdType].
 *
 * @param bannerType BannerAdType to use for the banner.
 * @param onAdResult Callback to invoke when the ad result is received.
 */
@Composable
fun TWMBannerComposable(
    bannerType: BannerAdType,
    onAdResult: (BannerAdResult) -> Unit = {}
) {

    val context = LocalContext.current

    val activity = remember(context) {
        generateSequence(context) {
            // keep traversing until we find the Activity
            when (it) {
                is ContextWrapper -> it.baseContext
                else -> null
            }
        }.filterIsInstance<Activity>()
            .firstOrNull()
    }

    if (activity == null)
        return

    val adView = remember {
        TWMAdView(activity, bannerType.adSize, bannerType.adUnitId).apply {
            adListener = object : TWMAdViewListener {
                override fun onReceiveAd(ad: TWMAd) {
                    onAdResult(BannerAdResult.OK)
                }

                override fun onFailedToReceiveAd(ad: TWMAd, errorCode: TWMAdRequest.ErrorCode) {
                    val errorResult = when (errorCode) {
                        TWMAdRequest.ErrorCode.NO_FILL -> BannerAdResult.Error("NO_FILL")
                        TWMAdRequest.ErrorCode.NETWORK_ERROR -> BannerAdResult.Error("NETWORK_ERROR")
                        TWMAdRequest.ErrorCode.INTERNAL_ERROR -> BannerAdResult.Error("INTERNAL_ERROR")
                        TWMAdRequest.ErrorCode.INVALID_REQUEST -> BannerAdResult.Error("INVALID_REQUEST")
                    }
                    onAdResult(errorResult)
                }

                override fun onPresentScreen(ad: TWMAd) {}
                override fun onDismissScreen(ad: TWMAd) {}
                override fun onLeaveApplication(ad: TWMAd) {}
            }

            // load ad immediately
            loadAd(TWMAdRequest())
        }
    }

    DisposableEffect(Unit) {
        onDispose {
            adView.destroy()
        }
    }

    AndroidView(
        factory = { adView }
    )
}
